﻿# PyU3D texture drawing example.
# Written by MysteriXYZ.

# Import the PyU3D package
import PyU3D as u3d
from PyU3D._local import *
import PyU3D.host as host

# Psyco is strongly recommended to boost the framerate
import psyco
psyco.full()

# Declare constants
SCREEN_WIDTH = 800
SCREEN_HEIGHT = 600
SPEED_LIMIT = 200


# Set U3D options

u3d.setLog("log.txt")
u3d.setZBufferFormat(32)
u3d.setMultiSampleType(0)


# initialize U3D and its host
host.init(
          SCREEN_WIDTH, SCREEN_HEIGHT, SPEED_LIMIT,
          "PyU3D texture drawing example",
          fullscreen=False
         )


# create an object to handle keyboard input
keyb = host.Keyboard()

# create an object to handle the mouse
mouse = host.Mouse()

# hide the cursor
mouse.setVisibility(False)

# should problems arise due to the cursor being hidden, try to use a transparent
# cursor instead of hiding it; comment out the line above and de-comment the
# line below
##mouse.setCursor(None)


# create a camera
camera = u3d.Camera(SCREEN_WIDTH, SCREEN_HEIGHT)

# set a font
arial12 = Font("Arial", 12.)
arial12.setColor(150, 75, 0)

# load texture
tex = Texture("Gfx/Subdivision.jpg", origin_x=.5, origin_y=.5)


def main():

  # Initialize variables for user control over texture drawing

  tex_angle = 0.
  tex_rotate = False
  tex_scale = 1.
  tex_left = 1.
  tex_width = 1.
  tex_origin = True

  # main loop
  while True:

    # process the current step of the game
    host.step()

    # Check user input

    if keyb.keyIsPressed("esc"):
      host.exit()
    elif keyb.keyIsPressed("kp-"):
      if keyb.keyIsDown("lshift") or keyb.keyIsDown("rshift"):
        tex_left -= .1
      elif keyb.keyIsDown("lctrl") or keyb.keyIsDown("rctrl"):
        tex_width -= .1
      else:
        tex.setOrigin(tex.getOrigin()[0]-.1)
    elif keyb.keyIsPressed("kp+"):
      if keyb.keyIsDown("lshift") or keyb.keyIsDown("rshift"):
        tex_left += .1
      elif keyb.keyIsDown("lctrl") or keyb.keyIsDown("rctrl"):
        tex_width += .1
      else:
        tex.setOrigin(tex.getOrigin()[0]+.1)
    elif keyb.keyIsPressed("kp/"):
      tex_scale -= .1
    elif keyb.keyIsPressed("kp*"):
      tex_scale += .1
    elif keyb.keyIsPressed("c"):
      tex.setOrigin(.5, .5)
    elif keyb.keyIsPressed("r"):
      tex_rotate = not tex_rotate
    elif keyb.keyIsPressed("o"):
      tex_origin = not tex_origin

    # update the transformation and projection of the camera
    camera.update()

    # increment the rotation angle of the texture, based on the actual speed of
    # the program
    if tex_rotate:
      tex_angle += 50. * host.getDeltaTime()

    # draw the texture
    tex.draw(
              x=SCREEN_WIDTH/2, y=SCREEN_HEIGHT/2,
              rot=tex_angle,
              scalx=tex_scale,
              scaly=1.,
              left=tex_left,
              width=tex_width,
              adjust_origin=tex_origin
            )

    # Display important statistics

    arial12.draw(10, 10, "fps: "+str(host.getFPS()))
    arial12.draw(10, 40, "triangles: "+str(getDrawnTriangleCount()))
    arial12.draw(10, 70, "draw calls: "+str(u3d.getDrawCallCount()))

    # Display control info

    arial12.draw(
                  10, SCREEN_HEIGHT-200,
                  "using origin (press <'o'> to toggle, <'c'> to centre) : "
                    +str(tex_origin)
                )
    arial12.draw(
                  10, SCREEN_HEIGHT-160,
                  "press <'r'> to toggle rotation"
                )
    arial12.draw(
                  10, SCREEN_HEIGHT-130,
                  "origin x (press keypad<->/<+> to change) : "
                    +str(tex.getOrigin()[0])
                )
    arial12.draw(
                  10, SCREEN_HEIGHT-100,
                  "horizontal scale (press keypad</>/<*> to change) : "
                    +str(tex_scale)
                )
    arial12.draw(
                  10, SCREEN_HEIGHT-70,
                  "left (press <Shift>+keypad<->/<+> to change) : "
                    +str(tex_left)
                )
    arial12.draw(
                  10, SCREEN_HEIGHT-40,
                  "width (press <Ctrl>+keypad<->/<+> to change) : "
                    +str(tex_width)
                )


if __name__ == '__main__':
  main()
